package hu.bme.mit.trainbenchmark.benchmark.runcomponents;

import java.io.File;
import java.io.IOException;

import org.apache.commons.exec.CommandLine;
import org.apache.commons.exec.DefaultExecutor;
import org.apache.commons.exec.ExecuteException;
import org.apache.commons.exec.ExecuteWatchdog;
import org.apache.commons.exec.Executor;
import org.apache.commons.exec.PumpStreamHandler;

import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableMap;
import hu.bme.mit.trainbenchmark.benchmark.config.BenchmarkConfig;
import hu.bme.mit.trainbenchmark.benchmark.memory.MemoryResult;
import hu.bme.mit.trainbenchmark.config.ExecutionConfig;
import org.apache.commons.exec.*;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Map;

public class BenchmarkRunner {

	public static String dictFormat(String format, Map<String, Object> values) {
		StringBuilder convFormat = new StringBuilder(format);
		ArrayList<Object> valueList = new ArrayList<>();
		int currentPos = 1;
		for (String key : values.keySet()) {
			String formatKey = "%(" + key + ")",
					formatPos = "%" + Integer.toString(currentPos) + "$1s";
			int index = -1;
			while ((index = convFormat.indexOf(formatKey, index)) != -1) {
				convFormat.replace(index, index + formatKey.length(), formatPos);
				index += formatPos.length();
			}
			valueList.add(values.get(key));
			++currentPos;
		}
		return String.format(convFormat.toString(), valueList.toArray());
	}

	public static int runPerformanceBenchmark(final BenchmarkConfig bc, final ExecutionConfig ec, String jvmSetting)
			throws IOException, InterruptedException {
		final Joiner joiner = Joiner.on(", ");
		System.out.println("Running benchmark.");
		System.out.println("Workload: " + bc.getConfigBase().getWorkload());
		System.out.println("Tool: " + bc.getToolName());
		System.out.println("Model: " + bc.getConfigBase().getModelPath());
		System.out.println("Description: " + bc.getDescription());
		System.out.println("Operations: [" + joiner.join(bc.getConfigBase().getOperations()) + "]");
		System.out.println("Execution configuration: " + ec);
		System.out.println("Runs: " + bc.getConfigBase().getRuns());

		final File configFile = File.createTempFile("trainbenchmark-benchmark-", ".conf");
		final String configPath = configFile.getAbsolutePath();
		bc.saveToFile(configPath);

		final String projectName = String.format("trainbenchmark-tool-%s", bc.getProjectName());
		final String jarPath = String.format("../%s/build/libs/%s-1.0.0-SNAPSHOT-fat.jar %s", projectName, projectName,
				configPath);

		final String javaCommand = dictFormat("java " + jvmSetting + " -jar %(jarPath) %(configPath)",
				ImmutableMap.of("Xms", ec.getXms(), "Xmx", ec.getXmx(),
						"jarPath", jarPath, "configPath", configPath));

//		final String javaCommand = String.format("java -Xms%s -Xmx%s -server -Xverify:none -jar %s %s", ec.getXms(), ec.getXmx(), jarPath, configPath);
//		final String javaCommand = String.format("java -Xms%s -Xmx%s -server -XX:+UnlockCommercialFeatures -XX:+FlightRecorder -XX:StartFlightRecording=dumponexit=true -Xverify:none -jar %s %s", ec.getXms(), ec.getXmx(), jarPath, configPath);
		final CommandLine cmdLine = CommandLine.parse(javaCommand);
		if (ec.getDryRun()) {
			System.out.println("Would have executed: "+ cmdLine.toString());
			return 0;
		}

		final long timeoutInSeconds = bc.getConfigBase().getTimeout();
		final long timeoutInMilliseconds = timeoutInSeconds * 1000;
		final ExecuteWatchdog watchdog = new ExecuteWatchdog(timeoutInMilliseconds);
		final Executor executor = new DefaultExecutor();
		executor.setWatchdog(watchdog);
		executor.setStreamHandler(new PumpStreamHandler());
		try {
			final int exitValue = executor.execute(cmdLine);
			System.out.println();
			return exitValue;
		} catch (final ExecuteException e) {
			if (watchdog.killedProcess()) {
				System.out.println("Process timed out.");
			} else {
				e.printStackTrace(System.out);
			}
			return e.getExitValue();
		}
	}

	public static int runMemoryBenchmark(final BenchmarkConfig bc, final ExecutionConfig defaultEc,
			final int numberOfSteps, String jvmSetting) throws IOException, InterruptedException {
		// e.g. initialMaxMemory = 12800, we save this (as a final variable), so
		// that we will not exceed it
		//
		// the memoryQuantum is halved on the start of each loop, so this starts
		// from 12800 as well (and will go down to 6400)
		final int initialMaxMemory = defaultEc.getMaxMemory();
		int currentMaxMemory = initialMaxMemory;
		int memoryQuantum = initialMaxMemory;
		int step = 0;

		while (step < numberOfSteps && currentMaxMemory <= initialMaxMemory) {
			step++;
			memoryQuantum /= 2;

			final ExecutionConfig ec = new ExecutionConfig(currentMaxMemory, currentMaxMemory);
			if (runPerformanceBenchmark(bc, ec, jvmSetting) == 0) {
				System.out.println("Execution finished, testing with less memory.");
				System.out.println();
				currentMaxMemory -= memoryQuantum;
			} else {
				System.out.println("Execution failed, testing with more memory.");
				System.out.println();
				currentMaxMemory += memoryQuantum;
			}
		}

		if (currentMaxMemory > initialMaxMemory) {
			System.out.println(
					"The first execution timed out or errored. Skipping larger sizes for this tool, scenario and query mix.");
			return -1;
		}
		System.out.println("Execution succeeded, estimated memory requirement: " + currentMaxMemory);
		System.out.println();

		final MemoryResult result = new MemoryResult(bc);
		final String memoryResultCsv = result.csvMemory(currentMaxMemory);
		result.serializeCsv(memoryResultCsv, "memory");

		return 0;
	}

}
