package de.tudresden.inf.st.jastadd.dumpAst.ast;

import net.sourceforge.plantuml.FileFormat;
import net.sourceforge.plantuml.FileFormatOption;
import net.sourceforge.plantuml.SourceStringReader;

import java.io.IOException;
import java.io.Writer;
import java.lang.reflect.InvocationTargetException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ResourceBundle;
import java.util.function.Consumer;
import java.util.function.Supplier;

import static de.tudresden.inf.st.jastadd.dumpAst.ast.ASTNode.matches;

/**
 * Building a dump.
 * <p>
 *
 * <h3>Inclusion and Exclusion of Types</h3>
 * Types can be only be disabled, see {@link #disableTypes(String, String...)}.
 *
 * <h3>Inclusion and Exclusion of children, tokens and relations</h3>
 * Children, tokens and relations are included by default.
 * This can be changed using exclusions and inclusion, both in general and per-type.
 * They are applied in the following order making later conditions take precedence over the first ones.
 * <ol>
 *   <li>Include everything as default.
 *   <li>Exclude general.
 *   <li>Include per type.
 *   <li>Exclude per type.
 * </ol>
 *
 * <h3>Inclusion and Exclusion of Attributes</h3>
 * Attributes are excluded by default, i.e., not shown.
 * This can be changed using inclusions and exclusions, both in general and per-type.
 * They are applied in the following order making later conditions take precedence over the first ones.
 * <ol>
 *   <li> Exclude everything as default.
 *   <li> Include general.
 *   <li> Exclude per type.
 *   <li> Include per type
 * </ol>
 */
@SuppressWarnings("UnusedReturnValue")
public class DumpBuilder {
  private final Object target;
  private String packageName;
  private DumpAst result;
  private final BuildConfig buildConfig;
  private final PrintConfig printConfig;

  protected DumpBuilder(Object target) {
    this.target = target;
    buildConfig = new BuildConfig();
    buildConfig.setIncludeChildMethod((parentNode, childNode, contextName) -> {
      // level 4: excluded for type? -> return no
      PatternCollection excludeOnType = buildConfig.matchExcludePatternCollection(parentNode.getClass().getSimpleName());
      if (excludeOnType != null && matches(excludeOnType.childPattern(), contextName)) {
        return false;
      }
      // level 3: included for type? -> return yes
      PatternCollection includeOnType = buildConfig.matchIncludePatternCollection(parentNode.getClass().getSimpleName());
      if (includeOnType != null && matches(includeOnType.childPattern(), contextName)) {
        return true;
      }
      // level 2: globally excluded? -> return no
      // level 1: otherwise return yes
      return !matches(buildConfig.getGlobalPatternCollection().childPattern(), contextName);
    });
    buildConfig.setIncludeRelationMethod((sourceNode, targetNode, roleName) -> {
      // level 4: excluded for type? -> return no
      PatternCollection excludeOnType = buildConfig.matchExcludePatternCollection(sourceNode.getClass().getSimpleName());
      if (excludeOnType != null && matches(excludeOnType.relationPattern(), roleName)) {
        return false;
      }
      // level 3: included for type? -> return yes
      PatternCollection includeOnType = buildConfig.matchIncludePatternCollection(sourceNode.getClass().getSimpleName());
      if (includeOnType != null && matches(includeOnType.relationPattern(), roleName)) {
        return true;
      }
      // level 2: globally excluded? -> return no
      // level 1: otherwise return yes
      return !matches(buildConfig.getGlobalPatternCollection().relationPattern(), roleName);
    });
    buildConfig.setIncludeTokenMethod((node, tokenName, value) -> {
      // level 4: excluded for type? -> return no
      PatternCollection excludeOnType = buildConfig.matchExcludePatternCollection(node.getClass().getSimpleName());
      if (excludeOnType != null && matches(excludeOnType.tokenPattern(), tokenName)) {
        return false;
      }
      // level 3: included for type? -> return yes
      PatternCollection includeOnType = buildConfig.matchIncludePatternCollection(node.getClass().getSimpleName());
      if (includeOnType != null && matches(includeOnType.tokenPattern(), tokenName)) {
        return true;
      }
      // level 2: globally excluded? -> return no
      // level 1: otherwise return yes
      return !matches(buildConfig.getGlobalPatternCollection().tokenPattern(), tokenName);
    });
    buildConfig.setIncludeAttributeMethod((node, attributeName, isNTA, supplier) -> {
      // level 4: included for type? -> return yes
      PatternCollection includeOnType = buildConfig.matchIncludePatternCollection(node.getClass().getSimpleName());
      if (includeOnType != null && matches(isNTA ? includeOnType.ntaPattern() : includeOnType.attributePattern(), attributeName)) {
        return true;
      }
      // level 3: excluded for type? -> return no
      PatternCollection excludeOnType = buildConfig.matchExcludePatternCollection(node.getClass().getSimpleName());
      if (excludeOnType != null && matches(isNTA ? excludeOnType.ntaPattern() : excludeOnType.attributePattern(), attributeName)) {
        return false;
      }
      // level 2: globally included? -> return yes
      // level 1: otherwise return no
      PatternCollection global = buildConfig.getGlobalPatternCollection();
      return matches(isNTA ? global.ntaPattern() : global.attributePattern(), attributeName);
    });
    buildConfig.setGlobalPatternCollection(new PatternCollection());
    buildConfig.setStyleInformation(StyleInformation.createDefault());
    printConfig = new PrintConfig();
    printConfig.setScale(1);
    printConfig.setVersion(readVersion());
  }

  /**
   * Add debug information in dumped content, mainly version numbers.
   * Also dump debugging information while reading in object.
   *
   * @return this
   */
  public DumpBuilder enableDebug() {
    buildConfig.setDebug(true);
    return this;
  }

  /**
   * Set the name of the package, where all AST classes are located.
   * @param packageName name of the package
   * @return this
   */
  public DumpBuilder setPackageName(String packageName) {
    this.packageName = packageName;
    return this;
  }

  /**
   * Include empty strings for all tokens
   *
   * @return this
   */
  public DumpBuilder includeEmptyStringsOnTokens() {
    buildConfig.setIncludeEmptyString(true);
    return this;
  }

  /**
   * Let all relations (and reference attributes) influence layouting of nodes (disabled by default).
   * @return this
   */
  public DumpBuilder enableRelationWithRank() {
    printConfig.setRelationWithRank(true);
    return this;
  }

  // --- Types ---

  /**
   * Disable all objects with types matching at least one of the given regex strings.
   * Disabled objects won't be included in any output. However, their children are still processed.
   * <p>
   * See {@link DumpBuilder} for details on inclusion and exclusion precedence.
   *
   * @param regex first pattern to match type names
   * @param moreRegexes more patterns to match type names
   * @return this
   * @see java.util.regex.Pattern#compile(String)
   */
  public DumpBuilder disableTypes(String regex, String... moreRegexes) {
    updateRegexes(buildConfig::getTypeIgnorePattern,
        buildConfig::setTypeIgnorePattern,
        regex, moreRegexes);
    return this;
  }

  // --- Tokens ---

  public <ASTNODE> DumpBuilder includeTokensWhen(IncludeTokenMethod<ASTNODE> spec) {
    buildConfig.setIncludeTokenMethod(spec);
    if (!buildConfig.getGlobalPatternCollection().getTokenPattern().isEmpty()) {
      System.err.println("Overriding previous filters for tokens");
    }
    return this;
  }

  /**
   * Exclude tokens and their value if the token name matches at least one of the given regex strings.
   * <p>
   * See {@link DumpBuilder} for details on inclusion and exclusion precedence.
   *
   * @param regex first pattern to match token names
   * @param moreRegexes more patterns to match token names
   * @return this
   * @see java.util.regex.Pattern#compile(String)
   */
  public DumpBuilder excludeTokens(String regex, String... moreRegexes) {
    updateRegexes(() -> buildConfig.getGlobalPatternCollection().getTokenPattern(),
        s -> buildConfig.getGlobalPatternCollection().setTokenPattern(s),
        regex, moreRegexes);
    return this;
  }

  /**
   * Exclude tokens and their value within a type if the token name matches at least one of the given regex strings.
   * <p>
   * See {@link DumpBuilder} for details on inclusion and exclusion precedence.
   *
   * @param typeRegex pattern to match a nonterminal name
   * @param regex first pattern to match token names
   * @param moreRegexes more patterns to match token names
   * @return this
   * @see java.util.regex.Pattern#compile(String)
   */
  public DumpBuilder excludeTokensFor(String typeRegex, String regex, String... moreRegexes) {
    PatternCollection collection = findOrCreateExcludePatternCollection(typeRegex);
    updateRegexes(collection::getTokenPattern,
        collection::setTokenPattern,
        regex, moreRegexes);
    return this;
  }

  /**
   * Include tokens (again) and their value within a type if the token name matches at least one of the given regex strings.
   * <p>
   * See {@link DumpBuilder} for details on inclusion and exclusion precedence.
   *
   * @param typeRegex pattern to match a nonterminal name
   * @param regex first pattern to match token names
   * @param moreRegexes more patterns to match token names
   * @return this
   * @see java.util.regex.Pattern#compile(String)
   */
  public DumpBuilder includeTokensFor(String typeRegex, String regex, String... moreRegexes) {
    PatternCollection collection = findOrCreateIncludePatternCollection(typeRegex);
    updateRegexes(collection::getTokenPattern,
        collection::setTokenPattern,
        regex, moreRegexes);
    return this;
  }

  // --- Children ---

  public <ASTNODE> DumpBuilder includeChildWhen(IncludeChildMethod<ASTNODE> spec) {
    buildConfig.setIncludeChildMethod(spec);
    if (!buildConfig.getGlobalPatternCollection().getChildPattern().isEmpty()) {
      System.err.println("Overriding previous filters for children");
    }
    return this;
  }

  /**
   * Exclude every child whose name (i.e., context) matches at least on of the given regex strings.
   * This means, that the complete object and its (transitive) children will never be included in any output.
   * <p>
   * See {@link DumpBuilder} for details on inclusion and exclusion precedence.
   *
   * @param regex first pattern to match child name
   * @param moreRegexes more patterns to match child names
   * @return this
   * @see java.util.regex.Pattern#compile(String)
   */
  public DumpBuilder excludeChildren(String regex, String... moreRegexes) {
    updateRegexes(() -> buildConfig.getGlobalPatternCollection().getChildPattern(),
        s -> buildConfig.getGlobalPatternCollection().setChildPattern(s),
        regex, moreRegexes);
    return this;
  }

  /**
   * Exclude every child within a type whose name (i.e., context) matches at least on of the given regex strings.
   * This means, that the complete object and its (transitive) children will never be included in any output.
   * <p>
   * See {@link DumpBuilder} for details on inclusion and exclusion precedence.
   *
   * @param typeRegex pattern to match a nonterminal name
   * @param regex first pattern to match child name
   * @param moreRegexes more patterns to match child names
   * @return this
   * @see java.util.regex.Pattern#compile(String)
   */
  public DumpBuilder excludeChildrenFor(String typeRegex, String regex, String... moreRegexes) {
    PatternCollection collection = findOrCreateExcludePatternCollection(typeRegex);
    updateRegexes(collection::getChildPattern,
        collection::setChildPattern,
        regex, moreRegexes);
    return this;
  }

  /**
   * Include every child (again) within a type whose name (i.e., context) matches at least on of the given regex strings.
   * This means, that the complete object and its (transitive) children will never be included in any output.
   * <p>
   * See {@link DumpBuilder} for details on inclusion and exclusion precedence.
   *
   * @param typeRegex pattern to match a nonterminal name
   * @param regex first pattern to match child name
   * @param moreRegexes more patterns to match child names
   * @return this
   * @see java.util.regex.Pattern#compile(String)
   */
  public DumpBuilder includeChildrenFor(String typeRegex, String regex, String... moreRegexes) {
    PatternCollection collection = findOrCreateIncludePatternCollection(typeRegex);
    updateRegexes(collection::getChildPattern,
        collection::setChildPattern,
        regex, moreRegexes);
    return this;
  }

  // --- Attributes ---

  public <ASTNODE> DumpBuilder includeAttributeWhen(IncludeAttributeMethod<ASTNODE> spec) {
    buildConfig.setIncludeAttributeMethod(spec);
    if (!buildConfig.getGlobalPatternCollection().getAttributePattern().isEmpty()) {
      System.err.println("Overriding previous filters for attributes");
    }
    return this;
  }

  /**
   * Include attributes (as tokens) and their value if the attribute name matches at least on of the given regex strings.
   * <p>
   * See {@link DumpBuilder} for details on inclusion and exclusion precedence.
   *
   * @param regex first pattern to match attribute name
   * @param moreRegexes more patterns to match attribute names
   * @return this
   * @see java.util.regex.Pattern#compile(String)
   */
  public DumpBuilder includeAttributes(String regex, String... moreRegexes) {
    updateRegexes(() -> buildConfig.getGlobalPatternCollection().getAttributePattern(),
        s -> buildConfig.getGlobalPatternCollection().setAttributePattern(s),
        regex, moreRegexes);
    return this;
  }

  /**
   * Include attributes within a type (as tokens) and their value if the attribute name matches at least on of the given regex strings.
   * <p>
   * See {@link DumpBuilder} for details on inclusion and exclusion precedence.
   *
   * @param typeRegex pattern to match a nonterminal name
   * @param regex first pattern to match attribute name
   * @param moreRegexes more patterns to match attribute names
   * @return this
   * @see java.util.regex.Pattern#compile(String)
   */
  public DumpBuilder includeAttributesFor(String typeRegex, String regex, String... moreRegexes) {
    PatternCollection collection = findOrCreateIncludePatternCollection(typeRegex);
    updateRegexes(collection::getAttributePattern,
        collection::setAttributePattern,
        regex, moreRegexes);
    return this;
  }

  /**
   * Exclude attributes within a type (as tokens) and their value (again) if the attribute name matches at least on of the given regex strings.
   * <p>
   * See {@link DumpBuilder} for details on inclusion and exclusion precedence.
   *
   * @param typeRegex pattern to match a nonterminal name
   * @param regex first pattern to match attribute name
   * @param moreRegexes more patterns to match attribute names
   * @return this
   * @see java.util.regex.Pattern#compile(String)
   */
  public DumpBuilder excludeAttributesFor(String typeRegex, String regex, String... moreRegexes) {
    PatternCollection collection = findOrCreateExcludePatternCollection(typeRegex);
    updateRegexes(collection::getAttributePattern,
        collection::setAttributePattern,
        regex, moreRegexes);
    return this;
  }

  // --- Nonterminal-Attributes ---

  /**
   * Includes nonterminal-attributes (as children) and their values if
   * their attribute name matches at least on of the given regex strings.
   * <br>
   * <b>Note</b>: A leading "get" and a trailing "List" in the name will be removed prior to matching.
   * Thus, it should not be contained in the regex either.
   * <p>
   * See {@link DumpBuilder} for details on inclusion and exclusion precedence.
   *
   * @param regex first pattern to match attribute name
   * @param moreRegexes more patterns to match attribute names
   * @return this
   * @see java.util.regex.Pattern#compile(String)
   */
  public DumpBuilder includeNonterminalAttributes(String regex, String... moreRegexes) {
    updateRegexes(() -> buildConfig.getGlobalPatternCollection().getNonterminalAttributePattern(),
        s -> buildConfig.getGlobalPatternCollection().setNonterminalAttributePattern(s),
        regex, moreRegexes);
    return this;
  }

  /**
   * Includes nonterminal-attributes (as children) and their values within a type if
   * their attribute name matches at least on of the given regex strings.
   * <br>
   * <b>Note</b>: A leading "get" and a trailing "List" in the name will be removed prior to matching.
   * Thus, it should not be contained in the regex either.
   * <p>
   * See {@link DumpBuilder} for details on inclusion and exclusion precedence.
   *
   * @param typeRegex pattern to match a nonterminal name
   * @param regex first pattern to match attribute name
   * @param moreRegexes more patterns to match attribute names
   * @return this
   * @see java.util.regex.Pattern#compile(String)
   */
  public DumpBuilder includeNonterminalAttributesFor(String typeRegex, String regex, String... moreRegexes) {
    PatternCollection collection = findOrCreateIncludePatternCollection(typeRegex);
    updateRegexes(collection::getNonterminalAttributePattern,
        collection::setNonterminalAttributePattern,
        regex, moreRegexes);
    return this;
  }

  /**
   * Excludes nonterminal-attributes (as children) and their values (again) within a type if
   * their attribute name matches at least on of the given regex strings.
   * <br>
   * <b>Note</b>: A leading "get" and a trailing "List" in the name will be removed prior to matching.
   * Thus, it should not be contained in the regex either.
   * <p>
   * See {@link DumpBuilder} for details on inclusion and exclusion precedence.
   *
   * @param typeRegex pattern to match a nonterminal name
   * @param regex first pattern to match attribute name
   * @param moreRegexes more patterns to match attribute names
   * @return this
   * @see java.util.regex.Pattern#compile(String)
   */
  public DumpBuilder excludeNonterminalAttributesFor(String typeRegex, String regex, String... moreRegexes) {
    PatternCollection collection = findOrCreateExcludePatternCollection(typeRegex);
    updateRegexes(collection::getNonterminalAttributePattern,
        collection::setNonterminalAttributePattern,
        regex, moreRegexes);
    return this;
  }

  // --- Relations ---

  public <ASTNODE> DumpBuilder includeRelationsWhen(IncludeRelationMethod<ASTNODE> spec) {
    buildConfig.setIncludeRelationMethod(spec);
    if (!buildConfig.getGlobalPatternCollection().getRelationPattern().isEmpty()) {
      System.err.println("Overriding previous filters for relations");
    }
    return this;
  }

  /**
   * Exclude every relation whose role-name matches at least on of the given regex strings.
   * This means two things: a) the relation to any potential target object(s) is never shown, and b) the target
   * object(s) are not shown unless they are reachable by another relation or by containment.
   * <p>
   * See {@link DumpBuilder} for details on inclusion and exclusion precedence.
   *
   * @param regex first pattern to match child name
   * @param moreRegexes more patterns to match child names
   * @return this
   * @see java.util.regex.Pattern#compile(String)
   */
  public DumpBuilder excludeRelations(String regex, String... moreRegexes) {
    updateRegexes(() -> buildConfig.getGlobalPatternCollection().getRelationPattern(),
        s -> buildConfig.getGlobalPatternCollection().setRelationPattern(s),
        regex, moreRegexes);
    return this;
  }

  /**
   * Exclude every relation within a type whose role-name matches at least on of the given regex strings.
   * This means two things: a) the relation to any potential target object(s) is never shown, and b) the target
   * object(s) are not shown unless they are reachable by another relation or by containment.
   * <p>
   * See {@link DumpBuilder} for details on inclusion and exclusion precedence.
   *
   * @param typeRegex pattern to match a nonterminal name
   * @param regex first pattern to match child name
   * @param moreRegexes more patterns to match child names
   * @return this
   * @see java.util.regex.Pattern#compile(String)
   */
  public DumpBuilder excludeRelationsFor(String typeRegex, String regex, String... moreRegexes) {
    PatternCollection collection = findOrCreateExcludePatternCollection(typeRegex);
    updateRegexes(collection::getRelationPattern,
        collection::setRelationPattern,
        regex, moreRegexes);
    return this;
  }

  /**
   * Include every relation (again) within a type whose role-name matches at least on of the given regex strings.
   * This means two things: a) the relation to any potential target object(s) is never shown, and b) the target
   * object(s) are not shown unless they are reachable by another relation or by containment.
   * <p>
   * See {@link DumpBuilder} for details on inclusion and exclusion precedence.
   *
   * @param typeRegex pattern to match a nonterminal name
   * @param regex first pattern to match child name
   * @param moreRegexes more patterns to match child names
   * @return this
   * @see java.util.regex.Pattern#compile(String)
   */
  public DumpBuilder includeRelationsFor(String typeRegex, String regex, String... moreRegexes) {
    PatternCollection collection = findOrCreateIncludePatternCollection(typeRegex);
    updateRegexes(collection::getRelationPattern,
        collection::setRelationPattern,
        regex, moreRegexes);
    return this;
  }

  // --- Settings ---


  /**
   * Omit children that are <code>null</code> (in a full AST, there are no such nodes).
   *
   * Normally, those nodes are shown.
   * @return this
   */
  public DumpBuilder excludeNullNodes() {
    buildConfig.setExcludeNullNodes(true);
    return this;
  }

  /**
   * Include children that are <code>null</code> (in a full AST, there are no such nodes).
   *
   * This is the default.
   * @return this
   */
  public DumpBuilder includeNullNodes() {
    buildConfig.setExcludeNullNodes(false);
    return this;
  }

  /**
   * Add custom preamble put before diagram content.
   * @param option arbitrary plantuml content
   * @return this
   */
  public DumpBuilder customPreamble(String option) {
    printConfig.addHeader(new Header(option));
    return this;
  }

  /**
   * Add skin param setting
   * @param setting the setting
   * @param value   value of the setting
   * @return this
   */
  public DumpBuilder skinParam(SkinParamStringSetting setting, String value) {
    customPreamble("skinparam " + setting.toString() + " " + value);
    return this;
  }

  /**
   * Add skin param setting
   * @param setting the setting
   * @param value   value of the setting
   * @return this
   */
  public DumpBuilder skinParam(SkinParamBooleanSetting setting, boolean value) {
    customPreamble("skinparam " + setting.toString() + " " + value);
    return this;
  }

  /**
   * Set the method defining, what name a node has (default: {@code n -> n == null ? "null" : n.getClass().getSimpleName() + "@" + Integer.toHexString(n.hashCode())}).
   *
   * <p>Example:<br>
   * {@code builder.<ASTNode<?>>setNameMethod(n -> n.isA() ? "A" : "Not A")}
   * @param nameMethod a style method
   * @param <ASTNODE> the type of ASTNode
   * @return this
   */
  public <ASTNODE> DumpBuilder setNameMethod(StyleMethod<ASTNODE> nameMethod) {
    buildConfig.getStyleInformation().setNameMethod(nameMethod);
    return this;
  }

  /**
   * Set the method defining, what background color a node has (default: {@code n -> ""}).
   *
   * <p>Example:<br>
   * {@code builder.<ASTNode<?>>setBackgroundColorMethod(n -> n.isA() ? "red" : "blue")}
   * @param colorMethod a style method
   * @param <ASTNODE> the type of ASTNode
   * @return this
   */
  public <ASTNODE> DumpBuilder setBackgroundColorMethod(StyleMethod<ASTNODE> colorMethod) {
    buildConfig.getStyleInformation().setBackgroundColorMethod(colorMethod);
    return this;
  }

  /**
   * Set the method defining, what text color a node has (default: {@code n -> ""}).
   *
   * <p>Example:<br>
   * {@code builder.<ASTNode<?>>setTextColorMethod(n -> n.isA() ? "black" : "white")}
   * @param colorMethod a style method
   * @param <ASTNODE> the type of ASTNode
   * @return this
   */
  public <ASTNODE> DumpBuilder setTextColorMethod(StyleMethod<ASTNODE> colorMethod) {
    buildConfig.getStyleInformation().setTextColorMethod(colorMethod);
    return this;
  }

  /**
   * Set the method defining, what stereotype a node has (default: {@code n -> ""}).
   *
   * <p>Example:<br>
   * {@code builder.<ASTNode<?>>setStereotypeMethod(n -> n.isA() ? "MyStereoType" : "")}
   * @param stereotypeMethod a style method
   * @param <ASTNODE> the type of ASTNode
   * @return this
   */
  public <ASTNODE> DumpBuilder setStereotypeMethod(StyleMethod<ASTNODE> stereotypeMethod) {
    buildConfig.getStyleInformation().setStereotypeMethod(stereotypeMethod);
    return this;
  }

  /**
   * Set color for computed parts of the AST (default: "blue").
   * @param color some color understood by plantuml
   * @return this
   * @see <a href="https://plantuml.com/en/color">https://plantuml.com/en/color</a>
   */
  public DumpBuilder setComputedColor(String color) {
    buildConfig.getStyleInformation().setComputedColor(color);
    return this;
  }

  /**
   * Set scale of diagram.
   * @param value the new scale (default: 1.0).
   * @return this
   */
  public DumpBuilder setScale(double value) {
    printConfig.setScale(value);
    return this;
  }

  /**
   * Enable enforced ordering of children nodes (may not always work due to dynamic placement by plantuml).
   * @return this
   */
  public DumpBuilder orderChildren() {
    printConfig.setOrderChildren(true);
    return this;
  }

  // --- Dump methods ---

  /**
   * Write out content as plantuml source code.
   *
   * @param destination path of destination file
   * @return this
   * @throws java.io.IOException if an I/O error happened during opening or writing in that file
   */
  public DumpBuilder dumpAsSource(Path destination) throws java.io.IOException {
    String content = build().toPlantUml();
    try (Writer writer = Files.newBufferedWriter(destination)) {
      writer.write(content);
    }
    return this;
  }

  /**
   * Write out content as intermediate data structure used by the template engine.
   *
   * @param destination path of destination file
   * @param prependCreationComment whether to add the creation date as first line
   * @return this
   * @throws java.io.IOException if an I/O error happened during opening or writing in that file
   */
  public DumpBuilder dumpAsYaml(Path destination, boolean prependCreationComment) throws java.io.IOException {
    String content = build().printYaml(prependCreationComment);
    try (Writer writer = Files.newBufferedWriter(destination)) {
      writer.write(content);
    }
    return this;
  }

  /**
   * Write out content as PNG image generated by plantuml.
   *
   * @param destination path of destination file
   * @return this
   * @throws java.io.IOException if an I/O error happened during opening or writing in that file
   */
  public DumpBuilder dumpAsPNG(Path destination) throws java.io.IOException {
    dumpAs(destination, FileFormat.PNG);
    return this;
  }

  /**
   * Write out content as SVG image generated by plantuml.
   *
   * @param destination path of destination file
   * @return this
   * @throws java.io.IOException if an I/O error happened during opening or writing in that file
   */
  public DumpBuilder dumpAsSVG(Path destination) throws java.io.IOException {
    dumpAs(destination, FileFormat.SVG);
    return this;
  }

  /**
   * Write out content as PDF generated by plantuml.
   *
   * <br>
   * <b>Note:</b> This requires additional dependencies, see <a href="https://plantuml.com/pdf">https://plantuml.com/pdf</a>
   *
   * @param destination path of destination file
   * @return this
   * @throws java.io.IOException if an I/O error happened during opening or writing in that file
   */
  public DumpBuilder dumpAsPDF(Path destination) throws java.io.IOException {
    dumpAs(destination, FileFormat.PDF);
    return this;
  }

  private void dumpAs(Path destination, FileFormat fileFormat) throws IOException {
    String content = build().toPlantUml();
    SourceStringReader reader = new SourceStringReader(content);
    reader.outputImage(Files.newOutputStream(destination), new FileFormatOption(fileFormat));
  }

  // --- Helper methods ---

  protected DumpAst build() {
    if (result == null) {
      result = new DumpAst();
      final String packageNameToUse;
      if (this.packageName != null) {
        packageNameToUse = this.packageName;
      } else {
        if (this.target == null) {
          packageNameToUse = null;
        } else {
          packageNameToUse = this.target.getClass().getPackage().getName();
        }
      }
      result.setPackageName(packageNameToUse);
      result.setBuildConfig(this.buildConfig);
      result.setPrintConfig(this.printConfig);
      try {
        result.transform(new TransformationTransferInformation(), this.target);
      } catch (InvocationTargetException | IllegalAccessException | NoSuchMethodException e) {
        throw new RuntimeException("Could not transform :(", e);
      }
    }
    return result;
  }

  private PatternCollection findOrCreateIncludePatternCollection(String typeRegex) {
    PatternCollection result = buildConfig.findIncludePatternCollection(typeRegex);
    if (result == null) {
      TypePatternCollectionMapping mapping = new TypePatternCollectionMapping();
      mapping.setTypeRegex(typeRegex);
      result = new PatternCollection();
      mapping.setPatternCollection(result);
      buildConfig.addIncludeTypePattern(mapping);
    }
    return result;
  }

  private PatternCollection findOrCreateExcludePatternCollection(String typeRegex) {
    PatternCollection result = buildConfig.findExcludePatternCollection(typeRegex);
    if (result == null) {
      TypePatternCollectionMapping mapping = new TypePatternCollectionMapping();
      mapping.setTypeRegex(typeRegex);
      result = new PatternCollection();
      mapping.setPatternCollection(result);
      buildConfig.addExcludeTypePattern(mapping);
    }
    return result;
  }

  private void updateRegexes(Supplier<String> getter, Consumer<String> setter, String regex, String... moreRegexes) {
    if (getter.get().isEmpty()) {
      setter.accept(regex);
    } else {
      setter.accept(getter.get() + "|" + regex);
    }
    for (String value : moreRegexes) {
      setter.accept(getter.get() + "|" + value);
    }
  }

  private String readVersion() {
    try {
      ResourceBundle resources = ResourceBundle.getBundle("dumpAstVersion");
      return resources.getString("version");
    } catch (java.util.MissingResourceException e) {
      return "version ?";
    }
  }
}
